<?php

namespace App\Http\Controllers\Api;

/*

Done by Nofalseo Software Services
nofalseo.com \ info@nofalseo.com

*/


use PDO;
use App\Models\Task;
use App\Models\TaskFile;
use Illuminate\Http\Request;
use App\Http\Helpers\HelperApp;
use App\Http\Helpers\HelperFile;
use Illuminate\Support\Facades\DB;
use App\Enums\NotificationTypeEnum;
use App\Http\Controllers\Controller;
use App\Http\Helpers\ResponseHelper;
use App\Http\Resources\TaskResource;
use App\Http\Requests\Api\TaskRequest;
use App\Http\Helpers\HelperNotification;
use App\Http\Requests\Api\TaskReportRequest;

class TaskController extends Controller
{

    public function index()
    {
        $tasks = Task::when(request('key_words'), function ($q) {
            $q->where('title', 'like', '%' . request('key_words') . '%');
        })->orderBy('id', 'desc')->paginate(config('app.paginate_number'));
        return ResponseHelper::sendResponseSuccess(TaskResource::collection($tasks)->response()->getData());
    }

    public function index_user()
    {
        $tasks = Task::when(request('key_words'), function ($q) {
            $q->where('title', 'like', '%' . request('key_words') . '%');
        })->where('to_user_id', auth('api')->id())->orderBy('id', 'desc')->paginate(config('app.paginate_number'));
        return ResponseHelper::sendResponseSuccess(TaskResource::collection($tasks)->response()->getData());
    }


    public function create_or_update_report(TaskReportRequest $request)
    {

        DB::beginTransaction();
        try {
            $task  = Task::where("id", $request->task_id)->where('to_user_id', auth('api')->id())->first();
            if (!$task) {
                return ResponseHelper::sendNotFound();
            }

            $task->update([
                "status" => "end",
                "description_report_task" => $request->description_report_task
            ]);

            $this->set_attachments($task, $request, "report");

            HelperNotification::set_users_ids([$task->created_by_id])
                ->set_message(["ar" => "تم الانتهاء من المهمة " . $task->title, "en" => "Task " . $task->title . " has been completed"])
                ->set_params($task->id)
                ->set_type(NotificationTypeEnum::TaskComplete->value)
                ->send_db()
                ->send_fcm()
                //  ->send_pusher()
                ->push_notification();

            DB::commit();
            return ResponseHelper::sendResponseSuccess();
        } catch (\Exception $e) {
            DB::rollBack();
            HelperApp::set_log_catch('create sub account', $e->getMessage());
            return ResponseHelper::sendBadRequest(__('messages.An error occurred in data entry'));
        }
    }


    public function delete_file_report(Request $request)
    {
        $file = TaskFile::where("id", $request->file_id)->whereHas('task', function ($q) {
            $q->where('to_user_id', auth('api')->id());
        })->where("type", "report")->first();
        if (!$file) {
            return ResponseHelper::sendNotFound();
        }

        HelperFile::delete($file->path);

        $file->delete();
        return ResponseHelper::sendResponseSuccess();
    }




    public function create(TaskRequest $request)
    {


        DB::beginTransaction();
        try {


            $task = Task::create($this->get_base_data($request));
            $this->set_attachments($task, $request, "task");
            HelperNotification::set_users_ids([$task->to_user_id])
                ->set_message(["ar" => "لديك مهمة جديدة " . $task->title, "en" => "You have a new task " . $task->title])
                ->set_params($task->id)
                ->set_type(NotificationTypeEnum::UserTask->value)
                ->send_db()
                ->send_fcm()
                //  ->send_pusher()
                ->push_notification();
            DB::commit();
            return ResponseHelper::sendResponseSuccess();
        } catch (\Exception $e) {
            DB::rollBack();
            HelperApp::set_log_catch('create task', $e->getMessage());
            return ResponseHelper::sendBadRequest(__('messages.An error occurred in data entry'));
        }
    }

    public function update(TaskRequest $request, $id)
    {


        DB::beginTransaction();
        try {

            $task = Task::where("id", $id)->first();
            if (!$task) {
                return ResponseHelper::sendNotFound();
            }
            if ($task->status == "end") {
                return ResponseHelper::sendBadRequest(__("messages.A submitted assignment cannot be modified"));
            }
            $task->update($this->get_base_data($request));
            $this->set_attachments($task, $request, "task");
            DB::commit();
            return ResponseHelper::sendResponseSuccess();
        } catch (\Exception $e) {
            DB::rollBack();
            HelperApp::set_log_catch('create task', $e->getMessage());
            return ResponseHelper::sendBadRequest(__('messages.An error occurred in data entry'));
        }
    }

    public function delete($id)
    {
        $task = Task::where("id", $id)->first();
        if (!$task) {
            return ResponseHelper::sendNotFound();
        }


        foreach ($task->files as $file) {
            HelperFile::delete($file->path);
        }

        $task->delete();
        return ResponseHelper::sendResponseSuccess();
    }


    public function delete_file(Request $request)
    {
        $file = TaskFile::where("id", $request->file_id)->where("type", "task")->first();
        if (!$file) {
            return ResponseHelper::sendNotFound();
        }

        HelperFile::delete($file->path);

        $file->delete();
        return ResponseHelper::sendResponseSuccess();
    }


    private function get_base_data($request)
    {
        return  [
            "created_by_id" => auth('api')->id(),
            "title" => $request->title,
            "description_task" => $request->description_task,
            "to_user_id" => $request->to_user_id,
            "date" => $request->date,
        ];
    }

    private function set_attachments($task, $request, $type)
    {
        $attachments_data = [];
        if ($request->attachments && count($request->attachments)) {
            foreach ($request->attachments as $attachment) {
                $attachments_data[] = [
                    "task_id" => $task->id,
                    "type" => $type,
                    "path" => HelperFile::upload($attachment, "tasks")['path'],
                ];
            }

            TaskFile::insert($attachments_data);
        }
    }
}
