<?php

namespace App\Http\Controllers\Api;

/*

Done by Nofalseo Software Services
nofalseo.com \ info@nofalseo.com

*/


use App\Http\Controllers\Controller;
use App\Http\Helpers\HelperApp;
use App\Http\Helpers\HelperFile;
use App\Http\Helpers\ResponseHelper;
use App\Http\Requests\Api\ProtocolRequest;
use App\Http\Resources\AnalysisPatientResource;
use App\Http\Resources\PatientResource;
use App\Http\Resources\ProtocolAnalysisGeneralResource;
use App\Http\Resources\ProtocolAnalysisTestGroupResource;
use App\Http\Resources\ProtocolResource;
use App\Http\Resources\TestGroupResource;
use App\Models\AnalysisPatient;
use App\Models\Medication;
use App\Models\Patient;
use App\Models\Protocol;
use App\Models\ProtocolMaleFactor;
use App\Models\ProtocolMedication;
use App\Models\ProtocolReportFile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ProtocolController extends Controller
{
    public function index_general()
    {

        $data = AnalysisPatient::where('doctor_id', auth('api')->id())->where("type", "general")->orderBy('id', 'desc')->when(request('key_words'), function ($q) {
            $q->whereHas("patient",  function ($q) {
                $q->where("name" , "like"  , "%".request('key_words')."%")
                ->orWhere("other_name" , "like"  , "%".request('key_words')."%")
                ->orWhere("other_medical_file_number" , "like"  , "%".request('key_words')."%")
                ->orWhere("medical_file_number" , "like"  , "%".request('key_words')."%");
            });
        })->with(['patient', 'doctor'])->paginate(config('app.paginate_number'));

        return ResponseHelper::sendResponseSuccess(ProtocolAnalysisGeneralResource::collection($data)->response()->getData());
    }


    public function index_test_group()
    {
        $data = AnalysisPatient::where('doctor_id', auth('api')->id())->where("type", "test_group")->orderBy('id', 'desc')->when(request('key_words'), function ($q) {
            $q->whereHas("patient",  function ($q) {
                $q->where("name" , "like"  , "%".request('key_words')."%")
                ->orWhere("other_name" , "like"  , "%".request('key_words')."%")
                ->orWhere("other_medical_file_number" , "like"  , "%".request('key_words')."%")
                ->orWhere("medical_file_number" , "like"  , "%".request('key_words')."%");
            });
        })->with(['patient', 'doctor'])->paginate(config('app.paginate_number'));

        return ResponseHelper::sendResponseSuccess(ProtocolAnalysisTestGroupResource::collection($data)->response()->getData());
    }

    public function history($patient_id)
    {
        $patient = Patient::where("id" , $patient_id)->first();
        if(!$patient){
            return ResponseHelper::sendNotFound();
        }
        $reports_patient = AnalysisPatient::with(['step_1', 'step_2', 'step_3', 'step_4_complete', 'step_5', 'step_6_freezings', 'step_6_thawings', 'step_7'])->where("type", "test_group")
       // ->where("doctor_id", auth('api')->id())
        ->where("patient_id", $patient_id)->get();

        $reports_patient_data = [];

        foreach ($reports_patient as $report) {
            foreach (['step_1', 'step_2', 'step_3', 'step_4_complete', 'step_5', 'step_6_freezings', 'step_6_thawings', 'step_7'] as $step)
                if ($report->$step) {
                    Log::info([
                        "step"=>$step,
                        "data"=>$report->$step->toArray()
                    ]);
                    // $reports_patient_data[] = [
                    //     "analysis_patient_id" => $report->id,
                    //     "step" => str_replace("_complete", "", $step),
                    //     "created_at"=>$report->$step->created_at
                    // ];


                    $reports_patient_data[] = [
                        "model_data"=>new TestGroupResource($report),
                        "analysis_patient_id" => $report->id,
                        "step" => str_replace("_complete", "", $step),
                    ];
                }
        }

        $statistics_controller = new StatisticsUserController();

        return ResponseHelper::sendResponseSuccess([
            "statistics"=>$statistics_controller->statistics($patient_id),
            "patient"=> new PatientResource($patient),
            "reports_patient"=>$reports_patient_data,

        ]);
    }



    public function show($id)
    {
        $analysis_ids_doctor = AnalysisPatient::where("doctor_id", auth("api")->id())->pluck("id")->toArray();
        $protocol = Protocol::where("id", $id)->whereIn("analysis_patient_id", $analysis_ids_doctor)->first();
        if (!$protocol) {
            return ResponseHelper::sendNotFound();
        }

        return ResponseHelper::sendResponseSuccess(new ProtocolResource($protocol));
    }


    public function create(ProtocolRequest $request)
    {
        DB::beginTransaction();
        try {


            if (!$this->check_analysis_to_doctor($request->analysis_patient_id)) {
                return ResponseHelper::sendNotFound();
            }
            $check_protcol = Protocol::where("analysis_patient_id", $request->analysis_patient_id)->first();
            if ($check_protcol) {
                return ResponseHelper::sendBadRequest(__("messages.You have entered the protocol For this analysis before"));
            }

            $protocol = Protocol::create($this->get_base_date($request));

            $this->set_data($request, $protocol);

            DB::commit();
            return ResponseHelper::sendResponseSuccess();
        } catch (\Exception $e) {
            DB::rollBack();
            HelperApp::set_log_catch('create protocol', $e->getMessage());
            return ResponseHelper::sendBadRequest(__('messages.An error occurred in data entry'));
        }
    }

    public function update(ProtocolRequest $request, $id)
    {

        DB::beginTransaction();
        try {
            if (!$this->check_analysis_to_doctor($request->analysis_patient_id)) {
                return ResponseHelper::sendNotFound();
            }

            $analysis_ids_doctor = AnalysisPatient::where("doctor_id", auth("api")->id())->pluck("id")->toArray();
            $protocol = Protocol::where("id", $id)->whereIn("analysis_patient_id", $analysis_ids_doctor)->first();
            if (!$protocol) {
                return ResponseHelper::sendNotFound();
            }

            $protocol->update($this->get_base_date($request));
            $this->set_data($request, $protocol);


            DB::commit();
            return ResponseHelper::sendResponseSuccess();
        } catch (\Exception $e) {
            DB::rollBack();
            HelperApp::set_log_catch('update protocol', $e->getMessage());
            return ResponseHelper::sendBadRequest(__('messages.An error occurred in data entry'));
        }
    }


    public function delete_file(Request $request)
    {
        $file = ProtocolReportFile::with('protocol')->where("id",  $request->file_id)->first();
        if (!$file || !$this->check_analysis_to_doctor($file->protocol->analysis_patient_id)) {
            return ResponseHelper::sendNotFound();
        }

        HelperFile::delete($file->path);
        $file->delete();
        return ResponseHelper::sendResponseSuccess();
    }





    private function get_base_date($request)
    {
        return [
            "analysis_patient_id" => $request->analysis_patient_id,
            "duration_of_infertility" => $request->duration_of_infertility,
            "number_of_children" => $request->number_of_children,
            "number_of_miscarriages" => $request->number_of_miscarriages,
            "previous_ivf_trials" => $request->previous_ivf_trials,
            "known_causes_of_infertility" => $request->known_causes_of_infertility,
            "female_factor" => $request->female_factor,
            "unexplained_infertility" => $request->unexplained_infertility,
            "report_female_factor" => $request->report_female_factor,
            "report_male_factor" => $request->report_male_factor,
            "report_other_factor" => $request->report_other_factor,
            "cycle_day" => $request->cycle_day,
            "date" => $request->date,
            "protocol" => $request->protocol,
            "report_ultrasound" => $request->report_ultrasound,
            "report_laparoscopy" => $request->report_laparoscopy,
            "report_hysteroscopy" => $request->report_hysteroscopy,
        ];
    }


    private function  check_analysis_to_doctor($analysis_id)
    {
        return AnalysisPatient::where('id', $analysis_id)->where('doctor_id', auth('api')->id())->exists();
    }


    private function set_data($request, $protocol)
    {
        ProtocolMedication::where("protocol_id", $protocol->id)->delete();
        ProtocolMaleFactor::where("protocol_id", $protocol->id)->delete();


        // start male factors
        if ($request->male_factor && count($request->male_factor)) {
            $factors = [];
            foreach ($request->male_factor as $factor) {
                $factors[] = [
                    "protocol_id" => $protocol->id,
                    "male_factor" => $factor,
                ];
            }
            ProtocolMaleFactor::insert($factors);
        }

        // end male factors

        // start ovarian stimulation

        if ($request->ovarian_stimulation && count($request->ovarian_stimulation)) {
            $ovarian_stimulation_data = [];
            foreach ($request->ovarian_stimulation as $ovarian_stimulation) {
                if (isset($ovarian_stimulation['medication_id']) && $ovarian_stimulation['medication_id']) {
                    $medication_id = $ovarian_stimulation['medication_id'];
                } else {
                    $medication = Medication::create([
                        "name" => $ovarian_stimulation['medication_name'],
                        "user_id" => auth('api')->id(),
                        "type" => "ovarian_stimulation"
                    ]);
                    $medication_id =  $medication->id;
                }

                $ovarian_stimulation_data[] = [
                    "protocol_id" => $protocol->id,
                    "medication_id" => $medication_id,
                    "dose" => $ovarian_stimulation['dose'],
                    "number_of_doses" => $ovarian_stimulation['number_of_doses'],
                    "report" => $ovarian_stimulation['report'],
                ];
            }
            ProtocolMedication::insert($ovarian_stimulation_data);
        }

        // end ovarian stimulation


        // start ovulation trigger

        if ($request->ovulation_trigger && count($request->ovulation_trigger)) {
            $ovulation_trigger_data = [];
            foreach ($request->ovulation_trigger as $ovulation_trigger) {
                if (isset($ovulation_trigger['medication_id']) && $ovulation_trigger['medication_id']) {
                    $medication_id = $ovulation_trigger['medication_id'];
                } else {
                    $medication = Medication::create([
                        "name" => $ovulation_trigger['medication_name'],
                        "user_id" => auth('api')->id(),
                        "type" => "ovulation_trigger"
                    ]);
                    $medication_id =  $medication->id;
                }

                $ovulation_trigger_data[] = [
                    "protocol_id" => $protocol->id,
                    "medication_id" => $medication_id,
                    "time" => $ovulation_trigger['time'],
                    "day" => $ovulation_trigger['day'],
                    "report" => $ovulation_trigger['report'],

                ];
            }
            ProtocolMedication::insert($ovulation_trigger_data);
        }

        // end ovulation trigger


        //start report files
        $report_files = [];

        foreach (['report_ultrasound', 'report_laparoscopy', 'report_hysteroscopy'] as $report) {
            $request_report_file = $report . "_file";
            if ($request->$request_report_file && count($request->$request_report_file)) {
                foreach ($request->$request_report_file as $file) {
                    $type = str_replace("report_",  "",  $report);
                    $report_files[] = [
                        "protocol_id" => $protocol->id,
                        "type" => $type,
                        "path" => HelperFile::upload($file, "protocol")['path'],
                    ];
                }
            }
        }
        if (count($report_files)) {
            ProtocolReportFile::insert($report_files);
        }
        // end report files

    }
}
