<?php

namespace App\Http\Controllers\Api;

/*

Done by Nofalseo Software Services
nofalseo.com \ info@nofalseo.com

*/

use App\Enums\UserRoleEnum;
use App\Http\Controllers\Controller;
use App\Http\Helpers\HelperApp;
use App\Http\Helpers\ResponseHelper;
use App\Http\Requests\Api\AnalysisPatientRequest;
use App\Http\Requests\Api\SampleReceivedRequest;
use App\Http\Requests\Api\SendReportGeneralRequest;
use App\Http\Resources\AnalysisPatientResource;
use App\Http\Resources\BookingPatientResource;
use App\Models\AnalysisPatient;
use App\Models\GeneralAnalysisVariable;
use App\Models\GroupGeneralAnalysis;
use App\Models\GroupGeneralAnalysisItem;
use App\Models\ReportGeneralAnalysis;
use App\Models\ReportGeneralAnalysisVariable;
use App\Models\User;
use App\Models\Variable;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Throwable;

class AnalysisPatientController extends Controller
{

    public function create(AnalysisPatientRequest $request)
    {



        if ($request->type == "general") {

            if (!$request->group_id && !$request->general_analysis_id) {
                return ResponseHelper::sendBadRequest(__("messages.Please select the group or analysis"));
            }
        }
        $data = $request->validated();
        if (isset($data['doctor_name']) && $data['doctor_name']) {
            $doctor = User::create([
                "name" => $data['doctor_name'],
                "type" => UserRoleEnum::Doctor,
                "is_new_from_system" => true
            ]);
            $data['doctor_id'] =  $doctor->id;
        }
        if ($request->type == "general") {
            unset($data['date'], $data['time']);
        } else {
            unset($data['general_analysis_id']);
        }

        if ($request->group_id && $request->type == "general") {

            $this->create_multi($request,  $request->group_id, $data['doctor_id']);
        } else {

            AnalysisPatient::create($data);
        }

        return ResponseHelper::sendResponseSuccess(null);
    }




    private function create_multi($request, $group_id, $doctor_id)
    {
        $data = [];
        $analysis_group =  GroupGeneralAnalysisItem::where('group_general_analysis_id', $group_id)->whereHas('general_analysis', function ($q) {
            $q->where('is_delete', 0);
        })->get()->pluck('general_analysis_id')->toArray();

        foreach ($analysis_group as $id) {
            $data[] = [
                "type" => "general",
                "patient_id" => $request->patient_id,
                "doctor_id" => $doctor_id,
                "general_analysis_id" => $id,
                "created_at" => Carbon::now(),
                "updated_at" => Carbon::now()
            ];
        }

        AnalysisPatient::insert($data);
    }


    public function general()
    {
        $data = AnalysisPatient::where("type", "general")->orderBy('id', 'desc')->when(request('key_words'), function ($q) {
            $q->whereHas("patient",  function ($q) {
                $q->where("name", "like", "%" . request('key_words') . "%")
                    ->orWhere("other_name", "like", "%" . request('key_words') . "%")
                    ->orWhere("other_medical_file_number", "like", "%" . request('key_words') . "%")
                    ->orWhere("medical_file_number", "like", "%" . request('key_words') . "%");
            });
        })->withoutGlobalScope('notype')->whereHas('general_analysis',  function ($q) {
            $q->whereNull('type');
        })->with(['patient', 'doctor'])->paginate(config('app.paginate_number'));
        return ResponseHelper::sendResponseSuccess(AnalysisPatientResource::collection($data)->response()->getData());
    }

    public function booking()
    {
        $data = AnalysisPatient::orderBy('id', 'desc')->when(request('key_words'), function ($q) {
            $key_words = request('key_words');
            $q->whereHas("patient",  function ($query) use ($key_words) {

                $query->where("name", "like", "%{$key_words}%")
                    ->orWhere("other_name", "like", "%{$key_words}%")
                    ->orWhere("age", $key_words)
                    ->orWhere("other_age", $key_words)
                    ->orWhere("gender", $key_words)
                    ->orWhere("medical_file_number", "like", "%{$key_words}%")
                    ->orWhere("other_medical_file_number", "like", "%{$key_words}%");
            });
        })->with(['patient', 'doctor']);
        if (request('from_date')) {
            $data = $data->where(function ($query) {
                $query->whereDate("created_at", ">=", request('from_date'));
            });
        }

        if (request('to_date')) {
            $data = $data->where(function ($query) {
                $query->whereDate("created_at", "<=", request('to_date'));
            });
        }
        $data = $data->paginate(config('app.paginate_number'));

        return ResponseHelper::sendResponseSuccess(BookingPatientResource::collection($data)->response()->getData());
    }

    public function delete_booking($id)
    {
        $analysis_patient = AnalysisPatient::find($id);
        if (!$analysis_patient) {
            return ResponseHelper::sendNotFound();
        }
        $analysis_patient->delete();
        return ResponseHelper::sendResponseSuccess();
    }




    public function sample_received(SampleReceivedRequest $request)
    {
        $analysis_patient = AnalysisPatient::find($request->analysis_patient_id);
        $analysis_patient->is_sample_received = true;
        $analysis_patient->save();
        return ResponseHelper::sendResponseSuccess(new AnalysisPatientResource($analysis_patient));
    }


    public function send_report(SendReportGeneralRequest $request)
    {
        DB::beginTransaction();
        try {

            $analysis_patient = AnalysisPatient::find($request->analysis_patient_id);
            $variables_analysis_ids  = GeneralAnalysisVariable::where('general_analysis_id',  $analysis_patient->general_analysis_id)->pluck("variable_id")->toArray();

            // التأكد من ان المتغيرات الخاصة بالتحليل مرسلة

            $variables_request_ids = collect($request->variables)->pluck("id")->toArray();

            foreach ($variables_analysis_ids as $id) {
                if (!in_array($id, $variables_request_ids)) {
                    $variable = Variable::find($id);
                    return ResponseHelper::sendResponseError(null, Response::HTTP_BAD_REQUEST, __("messages.You must enter the value") . " ( " . $variable->name . " )");
                }
            }

            // التأكد من القيم المرسلة جميعها خاصة بالتحليل
            foreach ($variables_request_ids as $id) {
                if (!in_array($id, $variables_analysis_ids)) {
                    $variable = Variable::find($id);
                    return ResponseHelper::sendResponseError(null, Response::HTTP_BAD_REQUEST,  " ( " . $variable->name . " ) " . __("messages.Not specific to analysis"));
                }
            }


            $main_data = [
                "analysis_patient_id" => $request->analysis_patient_id,
                "abstinecece_days" => $request->abstinecece_days,
                "user_id" => auth('api')->id(),
                "notes" => $request->notes,
            ];


            // التأكد من التقرير موجود من قبل
            $report = ReportGeneralAnalysis::where("analysis_patient_id", $request->analysis_patient_id)->first();
            if ($report) {
                $report->update($main_data);
                ReportGeneralAnalysisVariable::where("report_general_analysis_id", $report->id)->delete();
            } else {
                $report =  ReportGeneralAnalysis::create($main_data);
            }

            $report_variables = [];
            foreach ($request->variables as $item) {
                $report_variables[] = [
                    "report_general_analysis_id" => $report->id,
                    "variable_id" => $item['id'],
                    "value" => $item["value"],
                ];
            }

            ReportGeneralAnalysisVariable::insert($report_variables);
            DB::commit();
            return ResponseHelper::sendResponseSuccess(new AnalysisPatientResource($analysis_patient));
        } catch (Throwable $e) {
            DB::rollBack();

            HelperApp::set_log_catch("send report", $e->getMessage());
            return ResponseHelper::sendResponseError(null, Response::HTTP_BAD_REQUEST, __('messages.An error occurred in data entry'));
        }
    }
}
